#!/bin/ash
# Reprogram the IDT EEPROM

[ "$DEBUG" = "true" ] && set -x

: FILE: ENTERING $0 $1 $2 $3 $4

NORMAL_MODE=0x00
SAFE_MODE=0x01
FILE_TYPE=""
FILE_REV=""

BLINK=true
ETCPATH=/etc
BINPATH=/bin
LOGIT="logger -t $0 -p 6 -s "

func_setPath()
{
  SAFE_FILE="${ETCPATH}/safemode.bin"
  NORMAL_FILE="${ETCPATH}/normalmode.bin"

  PXIE_5AX="${ETCPATH}/m9505pxi.bin"
  PXIE_2AX="${ETCPATH}/m9502pxi.bin"
  EEPROM_CMD="${BINPATH}/eepromw"
}



cntTotal=0

func_stop_shelf()
{
  : Entering: func_stop_shelf $1 $2 $3 $4
  if $STOP_SHELF
  then
    echo "Stopping Shelfmanager..."
    2>/dev/null 1>/dev/null clia terminate
  fi
}

func_start_shelf()
{
  : Entering: func_start_shelf $1 $2 $3 $4
  if $STOP_SHELF
  then
    echo "Restarting Shelfmanager..."
    ( . /etc/profile && daemon -f shelfman 2>&1 > /dev/null )
  fi
}

func_error_exit()
{
  echo "$0 Exiting..."
  $LOGIT $*
  func_start_shelf
  exit 1
}



func_i2ca ()
{
  : Entering: func_i2ca $1 $2 $3 $4
  cnt=0
  notdone=true
  while $notdone
  do
     2>/dev/null i2ca $1 $2 $3 $4 $5 $6 $7 $8 $9 $10 $11 $12 $13 $14 $15

     if [ "$?" = "0" ]
     then
       notdone=false
     elif test $cnt != 10
     then
       cntTotal=`expr $cntTotal + 1`
       usleep 100
     else
       func_error_exit "$0: ERROR: Timed out ($cntTotal) attempting to communicate with $1 $2 $3 $4 $5" 
     fi

   cnt=`expr $cnt + 1`

  done

  
}

func_get_switch_mode()
{
  : Entering: func_get_switch_mode $1 $2 $3 $4

 # GPIO set for read mode?
 if [ "$(gpiomips func 4 | awk '{print $3}')" != "INPUT" ]
 then
   gpiomips config 4 1
 fi

 if [ "$(gpiomips get 4 | awk '{print $3}')" = "0" ]
 then 
   echo $NORMAL_MODE
 else
   echo $SAFE_MODE
 fi

}

# returns 0x00 or 0x01
func_get_eeprom_mode()
{
  : Entering: func_get_eeprom_mode $1 $2 $3 $4
#  func_read_eeprom 0x08 0x00
echo "NeedModeSupport"
}

# returns rev in format 0x00
func_get_eeprom_rev()
{
  : Entering: func_get_eeprom_rev $1 $2 $3 $4
#  func_read_eeprom 0x08 0x01
echo "NeedRevSupport"
}

func_i2ca_test ()
{ 
  : Entering: func_i2ca_test $1 $2 $3 $4
  # $1 is device address ie 0x72
  DEVICES=$(i2c_test | grep $1)
  if [ "$?" -eq 0 ]
  then
    return 
  else
    func_error_exit "$0: ERROR: Device $1 not found"
    exit
  fi
}

func_switchreset ()
{
  : Entering: func_switchreset $1 $2 $3 $4
  echo "Resetting IDT switch"

#  2>/dev/null func_i2ca 0 77 w 0x43 0x07 0x0F 0x0 0xF8 0x01 0x00 0x00 0x00 > /dev/null
  $EEPROM_CMD -i -idtreset
}


func_blink_on ()
{
  [ "$BLINK" = "true" ] && 2>/dev/null clia setfruledstate 10 0 1 BLINK 750 750 GREEN
}

func_blink_off ()
{
  [ "$BLINK" = "true" ] && 2>/dev/null clia setfruledstate 10 0 1 LOCAL GREEN
}

func_web_vars()
{
  : Entering: func_idt_webvars $*
  myswitch=$(func_get_switch_mode)
  if [ "$myswitch" = "$SAFE_MODE" ]
  then
    myFile=$SAFE_FILE
  else
    myFile=$NORMAL_FILE
  fi
  
  $EEPROM_CMD -i -idtimage $myFile -stats $*  | egrep "^z_"
}


func_idt_bootcheck()
{
  : Entering: func_idt_bootcheck $*
  
  myswitch=$(func_get_switch_mode)
  if [ "$myswitch" = "$SAFE_MODE" ]
  then
    myFile=$SAFE_FILE
    $LOGIT "ESM SW6.2 is off. PCIe Gen 1 safe mode enabled."
  else
    myFile=$NORMAL_FILE
    $LOGIT "ESM SW6.2 is on. PCIe Gen 2 default mode enabled."
  fi

#  func_blink_on
   $LOGIT "IDT EEPROM: Expecting Type: $myFile"  
  $EEPROM_CMD -i -idtimage $myFile -bootcheck $*
#  func_blink_off
}

func_pxie_bootcheck()
{
  : Entering: func_pxie_bootcheck $*
  chassis_type=$(cat /etc/chassis_no)
  case $chassis_type in
     F5AX) myFile=$PXIE_5AX
     ;;
     F2AX) myFile=$PXIE_2AX
     ;;
     *) myFile="UnknownChassis_$chassis_type"
     ;;
  esac

  chassis_sn=$(eepromr -c 1 A2 identity 1024 | grep "Chassis Serial Number" | awk '{print $5}')
  if [ "$chassis_sn" = "" ]
  then
    chassis_sn=$(eepromr -c 1 A4 identity 1024 | grep "Chassis Serial Number" | awk '{print $5}')
  fi

  if [ "$chassis_sn" = "" ]
  then
   func_error_exit "$0: Chassis Serial Number not found!!"
   exit 1
  fi

   $LOGIT "PXIe Backplane: Expecting Type: $chassis_type, $myFile, SN: $chassis_sn"  
#  func_blink_on
  $EEPROM_CMD -i -pxiimage $myFile -sn $chassis_sn -bootcheck $*
#  func_blink_off
}


#
## MAIN
#

# Verify access to devices
func_i2ca_test 71

###  2>/dev/null func_i2ca 0 71 w 1 > /dev/null

##func_i2ca_test 77

if [ "$1" = "-ns" ]
then
  shift
  STOP_SHELF=false
else
  STOP_SHELF=true
fi

if [ "$1" = "-install" ]
then
  shift
  LOGIT="logger -t $0 -p 6 -s INSTALL: "
  ETCPATH=.
  BINPATH=.
fi

func_setPath

func_stop_shelf

case "$1" in
     -idtbootcheck )
        shift
        func_idt_bootcheck $* 
     ;;
     
     -pxiebootcheck )
        shift
        func_pxie_bootcheck $*
     ;;
     
     -switchreset )
        shift
        func_switchreset   
     ;;
     
     -webvars )
       shift
       func_web_vars $*
     ;;

     -geteeprommode )
       mymode=$(func_get_eeprom_mode)
       echo "mymode= $mymode"
       [ "$mymode" = "$SAFE_MODE" ]   && echo "Loaded for Safemode"
       [ "$mymode" = "$NORMAL_MODE" ] && echo "Loaded for Normalmode"
     ;;


     * ) func_error_exit "Unknown option";;
esac

#2>/dev/null func_i2ca 0 71 w 0 > /dev/null

func_start_shelf
